{

    Unit CoverHelper

    Implements some useful functions dealing with covers.

    ---------------------------------------------------------------
    Covernator
    Copyright (C) 2010, Daniel Gaussmann
    http://www.gausi.de
    mail@gausi.de
    ---------------------------------------------------------------
    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA 02110, USA
    ---------------------------------------------------------------

}
unit CoverHelper;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics,
  Dialogs, StrUtils, ContNrs, Jpeg, PNGImage, AudioFileClass;


  procedure SucheCover(Pfad: String; CoverList: TStringList);
  function GetFrontCover(CoverList: TStrings): integer;

    // Converts data from a (id3tag)-picture-stream to a Bitmap.
  function PicStreamToImage(aStream: TStream; Mime: AnsiString; aBmp: TBitmap): Boolean;

  function GetCommonString(Strings: TStringList; tolerance: Integer; var Fehlstelle: Integer): String;
  procedure GetCoverInfos(AudioFileList: TObjectlist; out aArtist, aAlbum: String);

  function ResizeGraphicToStream(aGraphic: TGraphic; Mime: AnsiString; dest: TStream; W,h: Integer): boolean;
  procedure LoadGraphicIntoPicStream(aFilename: String; mime: AnsiString; stream: TMemoryStream);



implementation

function IsImageExt(aExt: String): boolean;
begin
    aExt := AnsiLowerCase(aExt);
    result := (aExt = '.jpg')
           OR (aExt = '.jpeg')
           OR (aExt = '.png')
           OR (aExt = '.bmp')
           OR (aExt = '.jfif');
end;


procedure SucheCover(pfad: String; CoverList:TStringList);
var sr : TSearchrec;
    dateityp:string;
begin
    pfad := IncludeTrailingPathDelimiter(pfad);

    if Findfirst(pfad+'*',FaAnyfile,sr) = 0 then
    repeat
      if (sr.name<>'.') AND (sr.name<>'..') then
      begin
          dateityp := ExtractFileExt(sr.Name);
          if IsImageExt(dateityp) then
              CoverList.Add(pfad + sr.Name);
      end;
    until Findnext(sr)<>0;
    Findclose(sr);
end;

function GetFrontCover(CoverList:TStrings):integer;
var i:integer;
begin
  result := -1;

  for i := 0 to CoverList.Count-1 do
    if AnsiContainsText(CoverList[i],'front') then
    begin
      result := i;
      break;
    end;

  if result = -1 then
    for i:=0 to CoverList.Count-1 do
      if AnsiContainsText(CoverList[i],'_a') then
      begin
        result := i;
        break;
      end;

  if result = -1 then
  begin
    for i:=0 to CoverList.Count-1 do
      if AnsiContainsText(CoverList[i],'folder') then
      begin
        result := i;
        break;
      end;
  end;

  if (result = -1) and (CoverList.Count > 0) then
      result := 0;
end;



function PicStreamToImage(aStream: TStream; Mime: AnsiString; aBmp: TBitmap): Boolean;
var jp: TJPEGImage;
    png: TPNGImage;
begin
    result := True;
    if (mime = 'image/jpeg') or (mime = 'image/jpg') or (AnsiUpperCase(String(Mime)) = 'JPG') then
    try
        aStream.Seek(0, soFromBeginning);
        jp := TJPEGImage.Create;
        try
          try
            jp.LoadFromStream(aStream);
            jp.DIBNeeded;
            aBmp.Assign(jp);
          except
            result := False;
            aBmp.Assign(NIL);
          end;
        finally
          jp.Free;
        end;
    except
        result := False;
        aBmp.Assign(NIL);
    end else
        if (mime = 'image/png') or (Uppercase(String(Mime)) = 'PNG') then
        try
            aStream.Seek(0, soFromBeginning);
            png := TPNGImage.Create;
            try
              try
                png.LoadFromStream(aStream);
                aBmp.Assign(png);
              except
                result := False;
                aBmp.Assign(NIL);
              end;
            finally
              png.Free;
            end;
        except
            result := False;
            aBmp.Assign(NIL);
        end else
        if (mime = 'image/bmp') or (Uppercase(String(Mime)) = 'BMP') then
            try
                aStream.Seek(0, soFromBeginning);
                aBmp.LoadFromStream(aStream);
            except
                result := False;
                aBmp.Assign(Nil);
            end else
                begin
                    aBmp.Assign(NIL);
                end;
end;

function SameString(string1, string2: String; tolerance: Integer; var Fehlstelle: Integer): Boolean;
var tmp: String;
    i, c: Integer;
begin
    // sicherstellen, dass Album2 mindestens so lang ist wie 1
    if length(string1) > length(string2) then
    begin
        tmp := string1;
        string1 := string2;
        string2 := tmp;
    end;
    c := 0;
    for i := 1 to length(string1) do
    begin
        if AnsiUpperCase(string1[i]) <> AnsiUpperCase(string2[i]) then
        begin
            inc(c);
            // erste Fehlerstelle zurckmelden!
            if Fehlstelle = 0 then Fehlstelle := i;
        end;
        if c > tolerance then break; // Abbruch. Zu viele Fehler
    end;
    result := c <= tolerance;
end;

function GetCommonString(Strings: TStringList; tolerance: Integer; var Fehlstelle: Integer): String;
var minlength, minidx, i: Integer;
    checkStr: String;
    durchlauf, durchlaufmax, errorCount: Integer;
begin
    durchlauf := 0;
    errorCount := 0;
    Fehlstelle := 0;
    CheckStr := '';
    case Strings.Count of
        0..1: durchlaufmax := 0;
        2..4: durchlaufmax := 1;
        5..9: durchlaufmax := 2
    else
        durchlaufmax := 3;
    end;

    if Strings.Count > 0 then
    repeat
          // Den String mit minimaler Lnge bestimmen
          minlength := length(Strings[0]);
          minidx := 0;
          for i := 1 to Strings.Count - 1 do
          begin
              if length(Strings[i]) < minlength then
              begin
                  minlength := length(Strings[i]);
                  minidx := i;
              end;
          end;
          checkStr := Strings[minidx];
          // diesen String aus der Liste entfernen, damit er beim nchsten Durchlauf nicht wieder gefunden wird
          Strings.Delete(minidx);

          // Den Rest der Stringliste berprfen, ob dieser String "passt"
          // ErrorCount zhlt dabei die Strings, die nicht passen
          errorCount := 0;
          Fehlstelle := 0;
          for i := 0 to Strings.Count - 1 do
          begin
              if not SameString(checkStr, Strings[i], tolerance, Fehlstelle) then
                  inc(errorCount);
              if errorCount > 1 then break;
          end;

          // Situation hier:
          // Wenn errorCount <= 1, dann ist CheckStr ein guter String fr die Liste
          // Wenn nicht, dann probieren wir das nochmal.
          inc(durchlauf);
    until (durchlauf > durchlaufmax) or (errorCount <= 1) or (strings.Count = 0);

    if ErrorCount <= 1 then
        result := CheckStr
    else
    begin
        if fehlstelle <= length(CheckStr) Div 2 +1 then
            result := ''
        else
        begin
            result := copy(CheckStr, 1, fehlstelle - 1) + ' ... ';
            fehlstelle := 0;
        end;
    end;
end;


procedure GetCoverInfos(AudioFileList: TObjectlist; out aArtist, aAlbum: String);
var str1: String;
    maxidx, i, fehlstelle: Integer;
    aStringlist: TStringList;
begin
    if AudioFileList.Count <= 25 then maxIdx := AudioFileList.Count-1 else maxIdx := 25;

    aStringlist := TStringList.Create;
    try
        for i := 0 to maxIdx do
            aStringlist.Add(TAudioFile(AudioFileList[i]).Artist);

        fehlstelle := 0;
        str1 := GetCommonString(aStringlist, 0, fehlstelle);  // bei Test auf "String gleich?" keinen Fehler zulassen
        if str1 = '' then
            aArtist := 'Various artists'
        else
            aArtist := str1; // Fehlstelle ist irrelevant

        aStringlist.Clear;
        // Dasselbe jetzt mit Album, aber mit Toleranz 1 bei den Strings
        for i := 0 to maxIdx do
            aStringlist.Add(TAudioFile(AudioFileList[i]).Album);

        fehlstelle := 0;
        str1 := GetCommonString(aStringlist, 1, fehlstelle);  // bei Test auf "String gleich?" einen Fehler zulassen  (cd1/2...)
        if str1 = '' then
            aAlbum := 'Unknown compilation'
        else
        begin
            if fehlstelle <= length(str1) Div 2 +1 then
                aAlbum := str1
            else
                aAlbum := copy(str1, 1, fehlstelle-1) + ' ... ';
        end;

    finally
        aStringlist.Free;
    end;
end;


function ResizeGraphicToStream(aGraphic: TGraphic; Mime: AnsiString; dest: TStream; W,h: Integer): boolean;
var BigBmp, SmallBmp: TBitmap;
    xfactor, yfactor:double;
    aJpg: TJpegImage;
    aPng: TPngImage;
begin
    result := True;
    BigBmp := TBitmap.Create;
    SmallBmp := TBitmap.Create;
    try
        SmallBmp.Width := W;
        SmallBmp.Height := H;

        if (aGraphic <> NIL) And ((aGraphic.Width > 0) And (aGraphic.Height > 0)) then
        begin
            BigBmp.Assign(aGraphic);

            xfactor:= (W) / BigBmp.Width;
            yfactor:= (H) / BigBmp.Height;
            if xfactor > yfactor then
              begin
                SmallBmp.Width := round(BigBmp.Width * yfactor);
                SmallBmp.Height := round(BigBmp.Height * yfactor);
              end else
              begin
                SmallBmp.Width := round(BigBmp.Width * xfactor);
                SmallBmp.Height := round(BigBmp.Height * xfactor);
              end;

            SetStretchBltMode(SmallBmp.Canvas.Handle, HALFTONE);
            StretchBlt(SmallBmp.Canvas.Handle, 0 ,0, SmallBmp.Width, SmallBmp.Height, BigBmp.Canvas.Handle, 0, 0, BigBmp.Width, BigBmp.Height, SRCCopy);

            if mime = 'image/png' then
            begin
                aPng := TPngImage.Create;
                try
                    aPng.Assign(Smallbmp);
                    dest.Size := 0;
                    aPng.SaveToStream(dest);
                finally
                    aPng.Free;
                end;
            end else
            begin
                aJpg := TJpegImage.Create;
                try
                    aJpg.CompressionQuality := 90;
                    aJpg.Assign(Smallbmp);
                    dest.Size := 0;
                    aJpg.SaveToStream(dest);
                finally
                    aJpg.Free;
                end;
            end;
        end;
    finally
        SmallBmp.Free;
        BigBmp.Free;
    end;
end;


procedure LoadGraphicIntoPicStream(aFilename: String; mime: AnsiString; stream: TMemoryStream);
var aJpg: TJpegImage;
    sourceGraphic: TPicture;
begin
    if mime = '' then
    begin
        sourceGraphic := TPicture.Create;
        try
            sourceGraphic.LoadFromFile(aFilename);
            // in jpeg konvertieren
            aJpg := TJpegImage.Create;
            try
                aJpg.CompressionQuality := 90;
                aJpg.Assign(sourceGraphic.Bitmap);
                stream.Size := 0;
                aJpg.SaveToStream(stream);
            finally
                aJpg.Free;
            end;
        finally
            sourceGraphic.Free;
        end;

    end else
        stream.LoadFromFile(aFilename);
end;

end.
